"""
keyword_planner_api.py

示例模板：使用 Google Ads API 的 Keyword Plan Idea 服务来获取关键词建议的示例（模板）。
注意：此文件仅为示例模板，不包含真实凭证。要运行此脚本，请按以下步骤操作：

1) pip install google-ads
2) 在项目根目录创建 google-ads.yaml 或使用环境变量配置凭证（示例格式见下方注释）。
3) 将 CUSTOMER_ID 替换为你的 Google Ads 客户 ID (格式：'123-456-7890')。
4) 根据需要调整 location_ids 与 language_id。

凭证示例（google-ads.yaml）:
developer_token: 'INSERT_DEVELOPER_TOKEN_HERE'
client_id: 'INSERT_OAUTH2_CLIENT_ID_HERE'
client_secret: 'INSERT_OAUTH2_CLIENT_SECRET_HERE'
refresh_token: 'INSERT_REFRESH_TOKEN_HERE'
login_customer_id: 'INSERT_MANAGER_ACCOUNT_CUSTOMER_ID_IF_NEEDED'  # 可选

更多说明请参考: https://developers.google.com/google-ads/api/docs/start
"""

from google.ads.googleads.client import GoogleAdsClient
from google.ads.googleads.errors import GoogleAdsException
import json
import sys

def get_keyword_ideas(client, customer_id, seed_keywords,
                      language_id='1000', location_ids=None, page_size=1000):
    """
    使用 KeywordPlanIdeaService 请求关键词建议。
    参数:
        client: 已配置的 GoogleAdsClient
        customer_id: 字符串，格式 '123-456-7890'
        seed_keywords: 列表，例如 ['seo', 'keyword research']
        language_id: 语言 ID（默认 1000 = English）。可根据需要更改。
        location_ids: 地理位置 ID 列表，例如 [2840]（美国）
    返回:
        suggestions: 列表 of dict，包含关键词文本、平均月搜索量等（视 API 自身返回字段）
    """
    try:
        service = client.get_service("KeywordPlanIdeaService")
        # 构建请求参数
        from google.ads.googleads.v14.services.types import GenerateKeywordIdeasRequest
        from google.ads.googleads.v14.common.types import KeywordSeed
        keyword_seed = KeywordSeed(keywords=seed_keywords)

        request = GenerateKeywordIdeasRequest(
            customer_id=customer_id.replace("-", ""),
            language=language_id,
            geo_target_constants=[f"geoTargetConstants/{lid}" for lid in (location_ids or [])],
            keyword_seed=keyword_seed,
            page_size=page_size
        )

        response = service.generate_keyword_ideas(request=request)

        suggestions = []
        for idea in response:
            ent = {
                "text": idea.text,
            }
            # 注意：不同 API 版本返回字段可能不同，下面示例为常见字段访问方式
            if hasattr(idea, 'keyword_idea_metrics') and idea.keyword_idea_metrics:
                metrics = idea.keyword_idea_metrics
                ent.update({
                    "avg_monthly_searches": getattr(metrics, "avg_monthly_searches", None),
                    "competition": getattr(metrics, "competition", None),
                    "competition_index": getattr(metrics, "competition_index", None)
                })
            suggestions.append(ent)

        return suggestions

    except GoogleAdsException as ex:
        print("Request failed with status %s and includes the following errors:" % ex.error.code().name)
        for error in ex.failure.errors:
            print("\tError with message: %s" % error.message)
        raise

if __name__ == '__main__':
    # 使用示例（本地运行请先配置 google-ads.yaml 或环境变量）
    try:
        client = GoogleAdsClient.load_from_storage(path='google-ads.yaml')  # or omit path to use env
    except Exception as e:
        print("无法加载 Google Ads 客户端配置：", e)
        sys.exit(1)

    CUSTOMER_ID = 'INSERT_CUSTOMER_ID_HERE'  # e.g. '123-456-7890'
    seed_keywords = ['seo', 'keyword research', '搜索引擎优化']

    suggestions = get_keyword_ideas(client, CUSTOMER_ID, seed_keywords, language_id='1000', location_ids=[])
    print(json.dumps(suggestions, ensure_ascii=False, indent=2))