"""
related_graph_generator.py

生成关键词共现/相关词交互式网络的示例脚本（使用 pyvis）。
运行此脚本将生成一个 HTML 文件（related_graph.html），可在浏览器中打开进行交互式查看。

依赖：pip install pyvis networkx
"""
from pyvis.network import Network
import networkx as nx

def build_graph_from_pairs(pairs):
    G = nx.Graph()
    for a, b in pairs:
        if G.has_edge(a, b):
            G[a][b]['weight'] += 1
        else:
            G.add_edge(a, b, weight=1)
    return G

def export_pyvis(G, out_html='related_graph.html', notebook=False):
    net = Network(height='800px', width='100%', bgcolor='#ffffff', font_color='black', notebook=notebook)
    net.force_atlas_2based()
    for node in G.nodes():
        net.add_node(node, label=node, title=str(node))
    for u, v, data in G.edges(data=True):
        weight = data.get('weight', 1)
        net.add_edge(u, v, value=weight, title=f'weight: {weight}')
    net.show(out_html)
    print(f'Interactive graph saved to {out_html}')

if __name__ == '__main__':
    # 示例数据：共现对
    pairs = [
        ('SEO', 'search engine'), ('SEO', 'keyword'), ('SEO', 'content'),
        ('keyword', 'search volume'), ('search engine', 'google'),
        ('content', 'blog'), ('content', 'copywriting'),
        ('google', 'bing'), ('seo', 'technical seo'), ('keyword', 'long tail keyword'),
        ('搜索引擎优化', '关键词研究'), ('搜索引擎优化', '内容优化'), ('关键词研究', '长尾关键词')
    ]
    G = build_graph_from_pairs(pairs)
    export_pyvis(G, out_html='related_graph.html')