# visibility_forecast.py
# 中文注释版：合并示例数据并用 Prophet 做短期预测
# 依赖：pip install pandas prophet openpyxl

import pandas as pd
import numpy as np
from prophet import Prophet
import os

# ---- 如果不存在示例数据，则生成小样例供演示 ----
if not os.path.exists("searchmetrics_visibility.csv") or not os.path.exists("gsc_weekly_clicks.csv"):
    print("未检测到示例 CSV，正在生成示例数据：searchmetrics_visibility.csv、gsc_weekly_clicks.csv")
    dates = pd.date_range(end=pd.Timestamp.today(), periods=60, freq='W-MON')
    np.random.seed(42)
    visibility = np.clip(100 + np.cumsum(np.random.normal(0, 2, len(dates))), 50, 200)
    clicks = np.clip(500 + np.cumsum(np.random.normal(0, 20, len(dates))), 100, 2000)
    sm_df = pd.DataFrame({"date": dates, "visibility_index": visibility})
    gsc_df = pd.DataFrame({"date": dates, "gsc_clicks": clicks})
    sm_df.to_csv("searchmetrics_visibility.csv", index=False, encoding="utf-8-sig")
    gsc_df.to_csv("gsc_weekly_clicks.csv", index=False, encoding="utf-8-sig")
    print("示例数据生成完毕。")

# 1) 读取 Searchmetrics 可见性
sm = pd.read_csv('searchmetrics_visibility.csv', parse_dates=['date'])
# 2) 读取 GSC 点击
gsc = pd.read_csv('gsc_weekly_clicks.csv', parse_dates=['date'])

# 3) 合并（按周）
df = pd.merge(sm, gsc, on='date', how='left').fillna(0)

# 4) 构建 Prophet 格式
df_prophet = df[['date','visibility_index']].rename(columns={'date':'ds','visibility_index':'y'})

# 5) 可选：添加外生变量示例（注释）
# df_prophet['ads_spend'] = np.random.randint(0,1000,len(df_prophet))
# m.add_regressor('ads_spend')

m = Prophet(weekly_seasonality=True, yearly_seasonality=True)
m.fit(df_prophet)

future = m.make_future_dataframe(periods=8, freq='W')
forecast = m.predict(future)

# 导出预测结果
forecast[['ds','yhat','yhat_lower','yhat_upper']].to_excel('visibility_forecast.xlsx', index=False)
print("✅ visibility_forecast.xlsx 已生成（含 yhat, yhat_lower, yhat_upper）")
