#!/usr/bin/env python3
"""Auto report generator combining video metadata and watch sessions.
Generates an Excel report and basic charts.

Usage:
  python auto_report.py --video_csv example_data/fetched_videos.csv --watch_csv example_data/sample_watch_sessions.csv --out report_weekly.xlsx
"""
import argparse, pandas as pd, os
import matplotlib.pyplot as plt

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--video_csv', required=True)
    parser.add_argument('--watch_csv', required=True)
    parser.add_argument('--out', default='report_weekly.xlsx')
    args = parser.parse_args()
    vids = pd.read_csv(args.video_csv)
    watch = pd.read_csv(args.watch_csv)
    merged = watch.merge(vids, on='videoId', how='left')
    # basic KPIs
    kpis = {
        'total_views': merged['watchSeconds'].sum(),
        'unique_videos': merged['videoId'].nunique(),
        'unique_users': merged['userId'].nunique()
    }
    summary = merged.groupby('videoId').agg(
        sessions=('sessionId','nunique'),
        total_watch_seconds=('watchSeconds','sum'),
        avg_watch_percent=('watchedPercent','mean')
    ).reset_index().sort_values('total_watch_seconds', ascending=False)
    os.makedirs(os.path.dirname(args.out) or '.', exist_ok=True)
    with pd.ExcelWriter(args.out, engine='openpyxl') as writer:
        pd.DataFrame([kpis]).to_excel(writer, sheet_name='KPIs', index=False)
        summary.to_excel(writer, sheet_name='VideoSummary', index=False)
        merged.to_excel(writer, sheet_name='Merged', index=False)
    # chart: top 8 videos by total watch seconds
    topn = summary.head(8)
    plt.figure(figsize=(10,6))
    plt.bar(topn['videoId'].astype(str), topn['total_watch_seconds'])
    plt.title('Top 8 Videos by Watch Seconds')
    plt.xlabel('Video ID')
    plt.ylabel('Total Watch Seconds')
    png = os.path.splitext(args.out)[0] + '_top8.png'
    plt.savefig(png)
    print('Report written to', args.out, 'and chart to', png)

if __name__ == '__main__':
    main()
