#!/usr/bin/env python3
"""Metadata generator template.
Example shows how to take video titles/descriptions and produce suggested title/description/tags.
Optionally integrates with OpenAI if OPENAI_API_KEY provided in config.

Usage:
  python metadata_generator.py --in example_data/fetched_videos.csv --out example_data/metadata_suggestions.csv --config config.json
"""
import argparse, csv, json, os
from collections import Counter
import re

def load_config(path):
    with open(path, 'r') as f:
        return json.load(f)

def simple_tag_suggester(text, top_n=8):
    text = re.sub(r'[^\w\s]', ' ', (text or '').lower())
    words = [w for w in text.split() if len(w)>2]
    cnt = Counter(words)
    common = [w for w,_ in cnt.most_common(top_n)]
    return common

def read_csv(path):
    with open(path, newline='', encoding='utf-8') as f:
        reader = csv.DictReader(f)
        return list(reader)

def write_csv(path, rows, fieldnames):
    with open(path, 'w', newline='', encoding='utf-8') as f:
        writer = csv.DictWriter(f, fieldnames=fieldnames)
        writer.writeheader()
        for r in rows:
            writer.writerow(r)

def generate(row):
    title = row.get('title','')
    desc = row.get('description','')
    suggest_title = title
    # simple heuristics: add key phrase if missing (placeholder)
    tags = simple_tag_suggester(title + ' ' + desc)
    # generate short SEO description (first 150 chars recommendation)
    seo_desc = (desc or '').strip().replace('\n',' ')[:300]
    return {
        'videoId': row.get('videoId'),
        'orig_title': title,
        'suggest_title': suggest_title,
        'seo_description': seo_desc,
        'suggest_tags': '|'.join(tags)
    }

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--in', dest='infile', required=True)
    parser.add_argument('--out', dest='outfile', default='metadata_suggestions.csv')
    parser.add_argument('--config', default='config.json')
    args = parser.parse_args()
    cfg = load_config(args.config)
    rows = read_csv(args.infile)
    out = []
    for r in rows:
        out.append(generate(r))
    fieldnames = ['videoId','orig_title','suggest_title','seo_description','suggest_tags']
    write_csv(args.outfile, out, fieldnames)
    print('Wrote metadata suggestions to', args.outfile)

if __name__ == '__main__':
    main()
