#!/usr/bin/env python3
"""Simple subtitle keyword extractor and SEO suggestions.
Usage:
  python subtitle_optimizer.py --srt example_data/sample_subtitles.srt --out example_data/subtitle_keywords.csv
"""
import argparse, re, csv
from collections import Counter

STOPWORDS = set([
 'the','and','for','that','this','with','you','are','have','but','not','was','they','from','what','when','where',
 'which','your','their','will','more','about','there','one','all','can','like','just','how','out','use','get'
])

def parse_srt(path):
    text = []
    with open(path, encoding='utf-8') as f:
        for line in f:
            line = line.strip()
            if not line or line.isdigit() or '-->' in line:
                continue
            text.append(line)
    return ' '.join(text)

def extract_keywords(text, top_n=20):
    words = re.sub(r'[^\w\s]', ' ', text.lower()).split()
    words = [w for w in words if len(w)>2 and w not in STOPWORDS]
    cnt = Counter(words)
    return cnt.most_common(top_n)

def write_csv(path, rows):
    with open(path, 'w', newline='', encoding='utf-8') as f:
        writer = csv.writer(f)
        writer.writerow(['keyword','count'])
        for k,c in rows:
            writer.writerow([k,c])

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--srt', required=True)
    parser.add_argument('--out', default='subtitle_keywords.csv')
    args = parser.parse_args()
    text = parse_srt(args.srt)
    kws = extract_keywords(text)
    write_csv(args.out, kws)
    print('Wrote', len(kws), 'keywords to', args.out)

if __name__ == '__main__':
    main()
