#!/usr/bin/env python3
"""YouTube Data API v3 fetcher.
Supports fetching recent videos from a channel and saving metadata to CSV.

Usage:
  python youtube_api_fetcher.py --config config.json --out fetched_videos.csv
"""
import argparse, json, csv, os, time
from googleapiclient.discovery import build
from googleapiclient.errors import HttpError

def load_config(path):
    with open(path, 'r') as f:
        return json.load(f)

def build_service(api_key=None):
    if api_key:
        return build('youtube', 'v3', developerKey=api_key)
    else:
        raise ValueError("API Key required for simple fetch in this template.")

def fetch_channel_videos(service, channel_id, max_results=50):
    results = []
    try:
        req = service.search().list(part='id,snippet', channelId=channel_id, maxResults=min(50, max_results), order='date', type='video')
        resp = req.execute()
        for item in resp.get('items', []):
            vid = item['id']['videoId']
            snip = item['snippet']
            results.append({
                'videoId': vid,
                'publishedAt': snip.get('publishedAt'),
                'title': snip.get('title'),
                'description': snip.get('description'),
                'channelTitle': snip.get('channelTitle')
            })
    except HttpError as e:
        print('YouTube API error:', e)
    return results

def fetch_video_stats(service, video_ids):
    stats = {}
    try:
        resp = service.videos().list(part='statistics,contentDetails,snippet', id=','.join(video_ids), maxResults=50).execute()
        for it in resp.get('items', []):
            vid = it['id']
            stats[vid] = {
                'viewCount': it.get('statistics', {}).get('viewCount'),
                'likeCount': it.get('statistics', {}).get('likeCount'),
                'commentCount': it.get('statistics', {}).get('commentCount'),
                'duration': it.get('contentDetails', {}).get('duration'),
                'tags': '|'.join(it.get('snippet', {}).get('tags', []))
            }
    except HttpError as e:
        print('YouTube API error:', e)
    return stats

def write_csv(path, rows, fieldnames):
    with open(path, 'w', newline='', encoding='utf-8') as f:
        writer = csv.DictWriter(f, fieldnames=fieldnames)
        writer.writeheader()
        for r in rows:
            writer.writerow(r)

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('--config', default='config.json')
    parser.add_argument('--out', default='fetched_videos.csv')
    parser.add_argument('--max', type=int, default=50)
    args = parser.parse_args()
    cfg = load_config(args.config)
    api_key = cfg.get('API_KEY')
    channel_id = cfg.get('CHANNEL_ID')
    service = build_service(api_key)
    items = fetch_channel_videos(service, channel_id, max_results=args.max)
    video_ids = [it['videoId'] for it in items]
    stats = {}
    if video_ids:
        stats = fetch_video_stats(service, video_ids)
    rows = []
    for it in items:
        vid = it['videoId']
        s = stats.get(vid, {})
        row = {
            'videoId': vid,
            'publishedAt': it.get('publishedAt'),
            'title': it.get('title'),
            'description': it.get('description'),
            'channelTitle': it.get('channelTitle'),
            'viewCount': s.get('viewCount'),
            'likeCount': s.get('likeCount'),
            'commentCount': s.get('commentCount'),
            'duration': s.get('duration'),
            'tags': s.get('tags')
        }
        rows.append(row)
    fieldnames = list(rows[0].keys()) if rows else ['videoId','title']
    os.makedirs(os.path.dirname(args.out) or '.', exist_ok=True)
    write_csv(args.out, rows, fieldnames)
    print('Saved', len(rows), 'videos to', args.out)

if __name__ == '__main__':
    main()
